# PHP Streaming Proxy System
## Professional Project Structure

```
proxy-system/
├── assets/
│   ├── css/
│   │   └── admin.css          # Admin panel styles
│   └── js/
│       └── admin.js           # Admin panel JavaScript
│
├── cache/
│   └── segments/              # Cached video segments
│
├── includes/
│   ├── functions.php          # Core utility functions
│   └── auth.php               # Authentication & session
│
├── config.php                 # Main configuration
├── database.sql               # Database schema
│
├── index.php                  # Landing page
├── login.php                  # Admin login
├── admin.php                  # Admin dashboard
├── stream.php                 # Stream proxy handler
├── player.php                 # Demo video player
│
├── debug.php                  # Debug tool (delete after use)
├── create_admin.php           # Admin creator (delete after use)
│
├── .htaccess                  # Apache configuration
├── .gitignore                 # Git ignore rules
└── README.md                  # Documentation
```

## 🎯 Features

### ✅ Modular Architecture
- **includes/functions.php** - URL handling, HTTP requests, caching, M3U8 processing
- **includes/auth.php** - Authentication, sessions, CSRF protection, rate limiting
- **assets/css/admin.css** - Modern dark theme, responsive design
- **assets/js/admin.js** - Copy functionality, form validation, auto-refresh

### ✅ Clean URLs
```
https://proxy.diz1.online/stream/TOKEN.m3u8
```

### ✅ Advanced Features
- Segment caching (5 minutes)
- CORS support
- SSL verification disabled (compatibility)
- Rate limiting
- CSRF protection
- Access logging
- Statistics dashboard

## 🚀 Installation

1. **Upload files** to your server
2. **Import database**: `database.sql`
3. **Configure**: Edit `config.php`
4. **Set permissions**: `chmod 755 cache/segments`
5. **Login**: `https://yourdomain.com/login.php`
   - Username: `admin`
   - Password: `admin123`

## 📖 Usage

### Add Stream
1. Login to admin panel
2. Enter stream name and M3U8 URL
3. Click "Stream Ekle"
4. Copy generated clean URL

### Test Stream
1. Use debug tool: `debug.php?token=YOUR_TOKEN`
2. Open stream URL in VLC or player
3. Use demo player: `player.php`

## 🔧 Configuration

Edit `config.php`:
```php
define('DB_HOST', 'localhost');
define('DB_USER', 'your_user');
define('DB_PASS', 'your_pass');
define('DB_NAME', 'your_db');
define('SITE_URL', 'https://yourdomain.com');
define('ENABLE_CACHE', true);
define('CACHE_LIFETIME', 300);
```

## 🎬 Integration

### HTML5 Video
```html
<video controls>
    <source src="https://yourdomain.com/stream/TOKEN.m3u8" 
            type="application/x-mpegURL">
</video>
```

### Video.js
```javascript
var player = videojs('player');
player.src({
    src: 'https://yourdomain.com/stream/TOKEN.m3u8',
    type: 'application/x-mpegURL'
});
```

## 🔐 Security

- ✅ Password hashing (bcrypt)
- ✅ CSRF protection
- ✅ Rate limiting
- ✅ SQL injection protection (PDO)
- ✅ XSS protection
- ✅ Session management

## 📊 Admin Panel

- Stream management (add/edit/delete)
- Statistics dashboard
- Access logs
- Cache management
- Token regeneration
- Clean URL generation

## 🐛 Debugging

Use `debug.php?token=YOUR_TOKEN` to check:
- Database connection
- Stream lookup
- Source M3U8 fetch
- Playlist parsing
- Proxy URL generation

**Delete debug.php after testing!**

## ⚠️ Important

1. Delete `create_admin.php` after first use
2. Delete `debug.php` after testing
3. Change default admin password
4. Use HTTPS in production
5. Set proper file permissions

## 📝 License

MIT License - Free to use and modify
