<?php
// Admin Dashboard
require_once 'config.php';
requireLogin();

$db = getDBConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add_stream') {
        $name = $_POST['name'] ?? '';
        $sourceUrl = $_POST['source_url'] ?? '';
        $streamType = $_POST['stream_type'] ?? 'm3u8';
        
        if (!empty($name) && !empty($sourceUrl)) {
            $token = generateToken();
            $stmt = $db->prepare("INSERT INTO streams (name, source_url, token, stream_type) VALUES (?, ?, ?, ?)");
            $stmt->execute([$name, $sourceUrl, $token, $streamType]);
            $success = 'Stream başarıyla eklendi!';
        }
    } elseif ($action === 'delete_stream') {
        $id = $_POST['id'] ?? 0;
        $stmt = $db->prepare("DELETE FROM streams WHERE id = ?");
        $stmt->execute([$id]);
        $success = 'Stream silindi!';
    } elseif ($action === 'toggle_stream') {
        $id = $_POST['id'] ?? 0;
        $stmt = $db->prepare("UPDATE streams SET is_active = NOT is_active WHERE id = ?");
        $stmt->execute([$id]);
        $success = 'Stream durumu güncellendi!';
    } elseif ($action === 'regenerate_token') {
        $id = $_POST['id'] ?? 0;
        $token = generateToken();
        $stmt = $db->prepare("UPDATE streams SET token = ? WHERE id = ?");
        $stmt->execute([$token, $id]);
        $success = 'Token yenilendi!';
    } elseif ($action === 'logout') {
        session_destroy();
        header('Location: login.php');
        exit;
    }
}

// Get all streams
$streams = $db->query("SELECT * FROM streams ORDER BY created_at DESC")->fetchAll();

// Get statistics
$totalStreams = count($streams);
$activeStreams = count(array_filter($streams, fn($s) => $s['is_active']));
$totalViews = $db->query("SELECT COUNT(*) as count FROM stream_logs")->fetch()['count'];

// Get pool statistics
$poolStats = StreamPool::getPoolStats();
$cacheSize = getCacheSize();
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - Streaming Proxy</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f7fa;
            color: #333;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 40px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 24px;
        }
        
        .header .user-info {
            display: flex;
            align-items: center;
            gap: 20px;
        }
        
        .logout-btn {
            background: rgba(255, 255, 255, 0.2);
            color: white;
            border: 1px solid rgba(255, 255, 255, 0.3);
            padding: 8px 20px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .logout-btn:hover {
            background: rgba(255, 255, 255, 0.3);
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 40px 20px;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
        }
        
        .stat-card h3 {
            color: #888;
            font-size: 14px;
            font-weight: 500;
            margin-bottom: 10px;
        }
        
        .stat-card .value {
            font-size: 36px;
            font-weight: 700;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .section {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .section h2 {
            margin-bottom: 20px;
            color: #333;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            color: #555;
            font-weight: 500;
        }
        
        input[type="text"],
        input[type="url"] {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
        }
        
        input:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .btn-danger {
            background: #e74c3c;
            color: white;
        }
        
        .btn-warning {
            background: #f39c12;
            color: white;
        }
        
        .btn-success {
            background: #27ae60;
            color: white;
        }
        
        .btn-sm {
            padding: 6px 12px;
            font-size: 14px;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        
        th {
            background: #f8f9fa;
            font-weight: 600;
            color: #555;
        }
        
        tr:hover {
            background: #f8f9fa;
        }
        
        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .status-active {
            background: #d4edda;
            color: #155724;
        }
        
        .status-inactive {
            background: #f8d7da;
            color: #721c24;
        }
        
        .stream-link {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .stream-link input {
            flex: 1;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            background: #f8f9fa;
        }
        
        .copy-btn {
            padding: 8px 16px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 14px;
        }
        
        .copy-btn:hover {
            background: #5568d3;
        }
        
        .actions {
            display: flex;
            gap: 8px;
        }
        
        .success {
            background: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            border-left: 4px solid #28a745;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🎬 Streaming Proxy Admin Panel</h1>
        <div class="user-info">
            <span>Hoş geldin, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="logout">
                <button type="submit" class="logout-btn">Çıkış</button>
            </form>
        </div>
    </div>
    
    <div class="container">
        <?php if (isset($success)): ?>
            <div class="success"><?php echo htmlspecialchars($success); ?></div>
        <?php endif; ?>
        
        <div class="stats">
            <div class="stat-card">
                <h3>Toplam Stream</h3>
                <div class="value"><?php echo $totalStreams; ?></div>
            </div>
            <div class="stat-card">
                <h3>Aktif Stream</h3>
                <div class="value"><?php echo $activeStreams; ?></div>
            </div>
            <div class="stat-card">
                <h3>Toplam Görüntüleme</h3>
                <div class="value"><?php echo $totalViews; ?></div>
            </div>
            <div class="stat-card">
                <h3>🔗 Eşzamanlı Kullanıcı</h3>
                <div class="value"><?php echo $poolStats['total_users']; ?></div>
                <small style="color: #888; font-size: 12px;">Max User Bypass Aktif</small>
            </div>
            <div class="stat-card">
                <h3>💾 Cache Boyutu</h3>
                <div class="value" style="font-size: 24px;"><?php echo formatBytes($cacheSize); ?></div>
            </div>
        </div>
        
        <div class="section">
            <h2>Yeni Stream Ekle</h2>
            <form method="POST">
                <input type="hidden" name="action" value="add_stream">
                <div class="form-group">
                    <label for="name">Stream Adı</label>
                    <input type="text" id="name" name="name" required placeholder="Örn: Kanal 1">
                </div>
                <div class="form-group">
                    <label for="stream_type">Stream Tipi</label>
                    <select id="stream_type" name="stream_type" style="width: 100%; padding: 12px 15px; border: 2px solid #e0e0e0; border-radius: 10px; font-size: 16px;">
                        <option value="m3u8">M3U8 (HLS)</option>
                        <option value="direct">Direkt Stream (TS/MP4/FLV)</option>
                    </select>
                    <small style="color: #888; margin-top: 5px; display: block;">
                        M3U8: .m3u8 uzantılı linkler için<br>
                        Direkt: .ts, .mp4, .flv veya port:2095 gibi direkt stream linkler için
                    </small>
                </div>
                <div class="form-group">
                    <label for="source_url">Kaynak URL</label>
                    <input type="url" id="source_url" name="source_url" required 
                           placeholder="https://example.com/stream/master.m3u8">
                </div>
                <button type="submit" class="btn btn-primary">Stream Ekle</button>
            </form>
        </div>
        
        <div class="section">
            <h2>Stream Listesi</h2>
            <?php if (empty($streams)): ?>
                <p>Henüz stream eklenmemiş.</p>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Ad</th>
                            <th>Durum</th>
                            <th>Proxy Link</th>
                            <th>İşlemler</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($streams as $stream): ?>
                            <tr>
                                <td><?php echo $stream['id']; ?></td>
                                <td><?php echo htmlspecialchars($stream['name']); ?></td>
                                <td>
                                    <span class="status-badge <?php echo $stream['is_active'] ? 'status-active' : 'status-inactive'; ?>">
                                        <?php echo $stream['is_active'] ? 'Aktif' : 'Pasif'; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="stream-link">
                                        <?php 
                                        $proxyUrl = SITE_URL;
                                        if ($stream['stream_type'] === 'direct') {
                                            $proxyUrl .= '/live/' . $stream['token'];
                                        } else {
                                            $proxyUrl .= '/stream/' . $stream['token'] . '.m3u8';
                                        }
                                        ?>
                                        <input type="text" readonly 
                                               value="<?php echo $proxyUrl; ?>" 
                                               id="link-<?php echo $stream['id']; ?>">
                                        <button class="copy-btn" onclick="copyLink(<?php echo $stream['id']; ?>)">Kopyala</button>
                                    </div>
                                    <small style="color: #888; font-size: 11px; margin-top: 5px; display: block;">
                                        Tip: <?php echo $stream['stream_type'] === 'direct' ? 'Direkt Stream' : 'M3U8 (HLS)'; ?>
                                    </small>
                                </td>
                                <td>
                                    <div class="actions">
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="toggle_stream">
                                            <input type="hidden" name="id" value="<?php echo $stream['id']; ?>">
                                            <button type="submit" class="btn btn-warning btn-sm">
                                                <?php echo $stream['is_active'] ? 'Pasif Yap' : 'Aktif Yap'; ?>
                                            </button>
                                        </form>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="regenerate_token">
                                            <input type="hidden" name="id" value="<?php echo $stream['id']; ?>">
                                            <button type="submit" class="btn btn-success btn-sm">Token Yenile</button>
                                        </form>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Silmek istediğinize emin misiniz?');">
                                            <input type="hidden" name="action" value="delete_stream">
                                            <input type="hidden" name="id" value="<?php echo $stream['id']; ?>">
                                            <button type="submit" class="btn btn-danger btn-sm">Sil</button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        function copyLink(id) {
            const input = document.getElementById('link-' + id);
            input.select();
            document.execCommand('copy');
            
            const btn = event.target;
            const originalText = btn.textContent;
            btn.textContent = 'Kopyalandı!';
            btn.style.background = '#27ae60';
            
            setTimeout(() => {
                btn.textContent = originalText;
                btn.style.background = '#667eea';
            }, 2000);
        }
    </script>
</body>
</html>
