// assets/js/admin.js
// Admin panel JavaScript functionality

document.addEventListener('DOMContentLoaded', function () {
    initCopyButtons();
    initDeleteConfirmations();
    initFormValidation();
});

// ============================================
// COPY TO CLIPBOARD
// ============================================

function initCopyButtons() {
    const copyButtons = document.querySelectorAll('.copy-btn');
    copyButtons.forEach(button => {
        button.addEventListener('click', function (e) {
            e.preventDefault();
            const streamId = this.getAttribute('data-id') || this.onclick.toString().match(/\d+/)[0];
            copyLink(streamId);
        });
    });
}

function copyLink(id) {
    const input = document.getElementById('link-' + id);
    if (!input) return;

    input.select();
    input.setSelectionRange(0, 99999); // For mobile

    try {
        document.execCommand('copy');
        showCopySuccess(event.target);
    } catch (err) {
        // Fallback to Clipboard API
        navigator.clipboard.writeText(input.value).then(() => {
            showCopySuccess(event.target);
        }).catch(err => {
            console.error('Failed to copy:', err);
        });
    }
}

function showCopySuccess(button) {
    const originalText = button.textContent;
    const originalBg = button.style.background;

    button.textContent = '✓ Kopyalandı!';
    button.style.background = '#27ae60';

    setTimeout(() => {
        button.textContent = originalText;
        button.style.background = originalBg;
    }, 2000);
}

// ============================================
// DELETE CONFIRMATIONS
// ============================================

function initDeleteConfirmations() {
    const deleteForms = document.querySelectorAll('form[data-confirm]');
    deleteForms.forEach(form => {
        form.addEventListener('submit', function (e) {
            const message = this.getAttribute('data-confirm') || 'Silmek istediğinize emin misiniz?';
            if (!confirm(message)) {
                e.preventDefault();
                return false;
            }
        });
    });
}

// ============================================
// FORM VALIDATION
// ============================================

function initFormValidation() {
    const forms = document.querySelectorAll('form[data-validate]');
    forms.forEach(form => {
        form.addEventListener('submit', function (e) {
            if (!validateForm(this)) {
                e.preventDefault();
                return false;
            }
        });
    });
}

function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('input[required], textarea[required]');

    inputs.forEach(input => {
        if (!input.value.trim()) {
            showError(input, 'Bu alan zorunludur');
            isValid = false;
        } else {
            clearError(input);
        }

        // URL validation
        if (input.type === 'url' && input.value) {
            if (!isValidUrl(input.value)) {
                showError(input, 'Geçerli bir URL girin');
                isValid = false;
            }
        }
    });

    return isValid;
}

function isValidUrl(string) {
    try {
        new URL(string);
        return true;
    } catch (_) {
        return false;
    }
}

function showError(input, message) {
    clearError(input);

    input.style.borderColor = '#e74c3c';
    const error = document.createElement('div');
    error.className = 'error-message';
    error.style.color = '#e74c3c';
    error.style.fontSize = '12px';
    error.style.marginTop = '5px';
    error.textContent = message;

    input.parentNode.appendChild(error);
}

function clearError(input) {
    input.style.borderColor = '';
    const error = input.parentNode.querySelector('.error-message');
    if (error) {
        error.remove();
    }
}

// ============================================
// STATS AUTO-REFRESH
// ============================================

function refreshStats() {
    fetch('api/stats.php')
        .then(response => response.json())
        .then(data => {
            updateStatCard('total-streams', data.total);
            updateStatCard('active-streams', data.active);
            updateStatCard('total-views', data.views);
        })
        .catch(err => console.error('Failed to refresh stats:', err));
}

function updateStatCard(id, value) {
    const element = document.getElementById(id);
    if (element) {
        element.textContent = value;
    }
}

// Refresh stats every 30 seconds
setInterval(refreshStats, 30000);

// ============================================
// UTILITY FUNCTIONS
// ============================================

function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type}`;
    notification.textContent = message;
    notification.style.position = 'fixed';
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.zIndex = '9999';
    notification.style.minWidth = '300px';
    notification.style.animation = 'slideIn 0.3s ease';

    document.body.appendChild(notification);

    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Add animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
