<?php
// Direct Stream Proxy Handler
// Handles direct TS/MP4/FLV streams (non-M3U8)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log');

require_once 'config.php';

// Parse request
$requestUri = $_SERVER['REQUEST_URI'];
$path = parse_url($requestUri, PHP_URL_PATH);

// Extract token from URL
preg_match('/\/(?:live|direct)\/([a-f0-9]+)/', $path, $matches);
$token = $matches[1] ?? '';

if (empty($token)) {
    http_response_code(400);
    header('Content-Type: text/plain');
    die('Error: Invalid token');
}

// Get stream
try {
    $stream = getStreamByToken($token);
    if (!$stream) {
        http_response_code(404);
        header('Content-Type: text/plain');
        die('Error: Stream not found or inactive');
    }
} catch (Exception $e) {
    error_log("Stream lookup error: " . $e->getMessage());
    http_response_code(500);
    header('Content-Type: text/plain');
    die('Error: Database error');
}

// Log access
try {
    if (getSetting('enable_logging', '1') == '1') {
        logStreamAccess($stream['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT'] ?? '');
    }
} catch (Exception $e) {
    error_log("Logging error: " . $e->getMessage());
}

// Stream the content
streamDirectContent($stream['source_url']);

function streamDirectContent($sourceUrl) {
    // Get content type
    $contentType = getContentType($sourceUrl);
    
    // Send headers
    header('Content-Type: ' . $contentType);
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, OPTIONS');
    header('Access-Control-Allow-Headers: *');
    header('Accept-Ranges: bytes');
    header('X-Proxy: Direct-Stream');
    
    // Handle range requests
    $range = $_SERVER['HTTP_RANGE'] ?? '';
    
    if (!empty($range)) {
        handleRangeRequest($sourceUrl, $range);
    } else {
        streamFullContent($sourceUrl);
    }
}

function handleRangeRequest($sourceUrl, $range) {
    // Get total size first
    $ch = curl_init($sourceUrl);
    curl_setopt_array($ch, [
        CURLOPT_NOBODY => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
    ]);
    
    curl_exec($ch);
    $totalSize = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
    curl_close($ch);
    
    // Parse range
    preg_match('/bytes=(\d+)-(\d*)/', $range, $matches);
    $start = intval($matches[1]);
    $end = !empty($matches[2]) ? intval($matches[2]) : ($totalSize > 0 ? $totalSize - 1 : 0);
    
    // Send partial content headers
    http_response_code(206);
    if ($totalSize > 0) {
        header("Content-Range: bytes $start-$end/$totalSize");
        header('Content-Length: ' . ($end - $start + 1));
    }
    
    // Stream range
    $ch = curl_init($sourceUrl);
    curl_setopt_array($ch, [
        CURLOPT_RANGE => "$start-$end",
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_BUFFERSIZE => 8192,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
        CURLOPT_HTTPHEADER => [
            'Accept: */*',
            'Connection: keep-alive'
        ],
        CURLOPT_WRITEFUNCTION => function($ch, $data) {
            echo $data;
            @flush();
            
            if (connection_aborted()) {
                return 0;
            }
            
            return strlen($data);
        }
    ]);
    
    curl_exec($ch);
    curl_close($ch);
    exit;
}

function streamFullContent($sourceUrl) {
    // Stream content in chunks
    $ch = curl_init($sourceUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_BUFFERSIZE => 8192,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
        CURLOPT_HTTPHEADER => [
            'Accept: */*',
            'Connection: keep-alive'
        ],
        CURLOPT_WRITEFUNCTION => function($ch, $data) {
            echo $data;
            @flush();
            
            // Check if client disconnected
            if (connection_aborted()) {
                return 0;
            }
            
            return strlen($data);
        }
    ]);
    
    $result = curl_exec($ch);
    $error = curl_error($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if (!$result && $error && !connection_aborted()) {
        error_log("Stream error: " . $error . " (HTTP $httpCode)");
    }
    
    exit;
}
