<?php
// includes/functions.php
// Core utility functions

// ============================================
// URL HELPERS
// ============================================

function makeAbsoluteUrl($url, $baseUrl, $currentUrl) {
    // Already absolute
    if (preg_match('/^https?:\/\//', $url)) {
        return $url;
    }
    
    // Protocol-relative URL
    if (strpos($url, '//') === 0) {
        $scheme = parse_url($currentUrl, PHP_URL_SCHEME);
        return $scheme . ':' . $url;
    }
    
    // Absolute path
    if ($url[0] === '/') {
        $parsed = parse_url($currentUrl);
        $base = $parsed['scheme'] . '://' . $parsed['host'];
        if (isset($parsed['port'])) {
            $base .= ':' . $parsed['port'];
        }
        return $base . $url;
    }
    
    // Relative path
    return $baseUrl . '/' . $url;
}

function getBaseUrl($url) {
    $parsedUrl = parse_url($url);
    $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    $baseUrl .= dirname($parsedUrl['path']);
    return $baseUrl;
}

// ============================================
// CONTENT TYPE DETECTION
// ============================================

function getContentType($url) {
    $ext = strtolower(pathinfo($url, PATHINFO_EXTENSION));
    
    $types = [
        'ts' => 'video/MP2T',
        'm3u8' => 'application/vnd.apple.mpegurl',
        'mp4' => 'video/mp4',
        'webm' => 'video/webm',
        'key' => 'application/octet-stream',
        'vtt' => 'text/vtt',
        'srt' => 'text/plain'
    ];
    
    return $types[$ext] ?? 'application/octet-stream';
}

// ============================================
// HTTP REQUESTS
// ============================================

function fetchUrl($url, $options = []) {
    $defaults = [
        'timeout' => 30,
        'connect_timeout' => 10,
        'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'follow_redirects' => true,
        'verify_ssl' => false
    ];
    
    $opts = array_merge($defaults, $options);
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => $opts['follow_redirects'],
        CURLOPT_SSL_VERIFYPEER => $opts['verify_ssl'],
        CURLOPT_SSL_VERIFYHOST => $opts['verify_ssl'] ? 2 : 0,
        CURLOPT_TIMEOUT => $opts['timeout'],
        CURLOPT_CONNECTTIMEOUT => $opts['connect_timeout'],
        CURLOPT_USERAGENT => $opts['user_agent'],
        CURLOPT_HTTPHEADER => [
            'Accept: */*',
            'Accept-Encoding: gzip, deflate',
            'Connection: keep-alive'
        ]
    ]);
    
    $content = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
    $error = curl_error($ch);
    curl_close($ch);
    
    return [
        'success' => ($httpCode === 200),
        'http_code' => $httpCode,
        'content' => $content,
        'content_type' => $contentType,
        'error' => $error
    ];
}

// ============================================
// CACHE MANAGEMENT
// ============================================

function getCachedContent($key, $maxAge = CACHE_LIFETIME) {
    if (!ENABLE_CACHE) {
        return false;
    }
    
    $cacheFile = SEGMENT_CACHE_DIR . '/' . md5($key);
    
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < $maxAge) {
        return file_get_contents($cacheFile);
    }
    
    return false;
}

function setCachedContent($key, $content) {
    if (!ENABLE_CACHE || empty($content)) {
        return false;
    }
    
    $cacheFile = SEGMENT_CACHE_DIR . '/' . md5($key);
    return @file_put_contents($cacheFile, $content);
}

function clearCache() {
    if (!ENABLE_CACHE) {
        return false;
    }
    
    $files = glob(SEGMENT_CACHE_DIR . '/*');
    foreach ($files as $file) {
        if (is_file($file)) {
            @unlink($file);
        }
    }
    return true;
}

function getCacheSize() {
    if (!ENABLE_CACHE) {
        return 0;
    }
    
    $size = 0;
    $files = glob(SEGMENT_CACHE_DIR . '/*');
    foreach ($files as $file) {
        if (is_file($file)) {
            $size += filesize($file);
        }
    }
    return $size;
}

function formatBytes($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}

// ============================================
// DATABASE HELPERS
// ============================================

function logStreamAccess($streamId, $ip, $userAgent) {
    try {
        $db = getDBConnection();
        $stmt = $db->prepare("INSERT INTO stream_logs (stream_id, ip_address, user_agent) VALUES (?, ?, ?)");
        $stmt->execute([$streamId, $ip, $userAgent]);
        return true;
    } catch (Exception $e) {
        error_log("Failed to log stream access: " . $e->getMessage());
        return false;
    }
}

function getSetting($key, $default = null) {
    try {
        $db = getDBConnection();
        $stmt = $db->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        return $result ? $result['setting_value'] : $default;
    } catch (Exception $e) {
        error_log("Failed to get setting: " . $e->getMessage());
        return $default;
    }
}

function setSetting($key, $value) {
    try {
        $db = getDBConnection();
        $stmt = $db->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute([$key, $value, $value]);
        return true;
    } catch (Exception $e) {
        error_log("Failed to set setting: " . $e->getMessage());
        return false;
    }
}

function getStreamByToken($token) {
    try {
        $db = getDBConnection();
        $stmt = $db->prepare("SELECT * FROM streams WHERE token = ? AND is_active = 1");
        $stmt->execute([$token]);
        return $stmt->fetch();
    } catch (Exception $e) {
        error_log("Failed to get stream: " . $e->getMessage());
        return false;
    }
}

function getAllStreams() {
    try {
        $db = getDBConnection();
        return $db->query("SELECT * FROM streams ORDER BY created_at DESC")->fetchAll();
    } catch (Exception $e) {
        error_log("Failed to get streams: " . $e->getMessage());
        return [];
    }
}

function getStreamStats() {
    try {
        $db = getDBConnection();
        
        $total = $db->query("SELECT COUNT(*) as count FROM streams")->fetch()['count'];
        $active = $db->query("SELECT COUNT(*) as count FROM streams WHERE is_active = 1")->fetch()['count'];
        $views = $db->query("SELECT COUNT(*) as count FROM stream_logs")->fetch()['count'];
        
        return [
            'total' => $total,
            'active' => $active,
            'views' => $views
        ];
    } catch (Exception $e) {
        error_log("Failed to get stats: " . $e->getMessage());
        return ['total' => 0, 'active' => 0, 'views' => 0];
    }
}

// ============================================
// M3U8 PROCESSING
// ============================================

function processM3U8Playlist($content, $sourceUrl, $token, $type = 'master') {
    $baseUrl = getBaseUrl($sourceUrl);
    $lines = explode("\n", $content);
    $output = '';
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        // Keep comments and empty lines
        if (empty($line) || $line[0] === '#') {
            $output .= $line . "\n";
            continue;
        }
        
        // Process URLs
        $fullUrl = makeAbsoluteUrl($line, $baseUrl, $sourceUrl);
        
        // Determine if it's a playlist or segment
        if (stripos($line, '.m3u8') !== false) {
            // Variant playlist
            $proxyUrl = SITE_URL . '/stream/' . $token . '.m3u8?type=variant&url=' . urlencode($fullUrl);
        } else {
            // Segment
            $proxyUrl = SITE_URL . '/stream/' . $token . '.m3u8?type=segment&segment=' . urlencode($fullUrl);
        }
        
        $output .= $proxyUrl . "\n";
    }
    
    return $output;
}

// ============================================
// RESPONSE HELPERS
// ============================================

function sendM3U8Response($content) {
    header('Content-Type: application/vnd.apple.mpegurl');
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, OPTIONS');
    header('Access-Control-Allow-Headers: *');
    header('Cache-Control: no-cache, no-store, must-revalidate');
    header('Pragma: no-cache');
    header('Expires: 0');
    echo $content;
    exit;
}

function sendSegmentResponse($content, $contentType, $cached = false) {
    header('Content-Type: ' . $contentType);
    header('Access-Control-Allow-Origin: *');
    header('X-Cache: ' . ($cached ? 'HIT' : 'MISS'));
    header('Cache-Control: public, max-age=300');
    header('Content-Length: ' . strlen($content));
    echo $content;
    exit;
}

function sendError($code, $message) {
    http_response_code($code);
    header('Content-Type: application/json');
    echo json_encode(['error' => $message, 'code' => $code]);
    exit;
}
