<?php
// includes/stream_pool.php
// Connection pooling to bypass max user limits

class StreamPool {
    private static $pools = [];
    private static $poolDir = null;
    
    public static function init() {
        self::$poolDir = BASE_PATH . '/cache/pools';
        if (!file_exists(self::$poolDir)) {
            mkdir(self::$poolDir, 0755, true);
        }
    }
    
    // Get or create a shared stream connection
    public static function getSharedStream($streamId, $sourceUrl, $type = 'master') {
        self::init();
        
        $poolKey = md5($streamId . '_' . $type . '_' . $sourceUrl);
        $poolFile = self::$poolDir . '/' . $poolKey;
        $lockFile = $poolFile . '.lock';
        
        // Check if there's an active pool for this stream
        if (file_exists($poolFile)) {
            $poolData = json_decode(file_get_contents($poolFile), true);
            
            // Check if pool is still fresh (within 10 seconds)
            if ($poolData && (time() - $poolData['timestamp']) < 10) {
                // Increment user count
                $poolData['users']++;
                $poolData['last_access'] = time();
                file_put_contents($poolFile, json_encode($poolData));
                
                return [
                    'pooled' => true,
                    'content' => $poolData['content'],
                    'users' => $poolData['users']
                ];
            }
        }
        
        // Create new pool or refresh existing
        return self::createPool($poolKey, $poolFile, $sourceUrl);
    }
    
    private static function createPool($poolKey, $poolFile, $sourceUrl) {
        // Fetch fresh content
        $result = fetchUrl($sourceUrl);
        
        if (!$result['success']) {
            return [
                'pooled' => false,
                'error' => 'Failed to fetch source'
            ];
        }
        
        // Create pool data
        $poolData = [
            'timestamp' => time(),
            'last_access' => time(),
            'users' => 1,
            'content' => $result['content'],
            'source_url' => $sourceUrl
        ];
        
        // Save pool
        file_put_contents($poolFile, json_encode($poolData));
        
        return [
            'pooled' => true,
            'content' => $result['content'],
            'users' => 1,
            'fresh' => true
        ];
    }
    
    // Release a stream connection
    public static function releaseStream($streamId, $sourceUrl, $type = 'master') {
        self::init();
        
        $poolKey = md5($streamId . '_' . $type . '_' . $sourceUrl);
        $poolFile = self::$poolDir . '/' . $poolKey;
        
        if (file_exists($poolFile)) {
            $poolData = json_decode(file_get_contents($poolFile), true);
            
            if ($poolData) {
                $poolData['users'] = max(0, $poolData['users'] - 1);
                
                // If no users, delete pool after 30 seconds
                if ($poolData['users'] == 0) {
                    $poolData['expire_at'] = time() + 30;
                }
                
                file_put_contents($poolFile, json_encode($poolData));
            }
        }
    }
    
    // Clean old pools
    public static function cleanOldPools() {
        self::init();
        
        $files = glob(self::$poolDir . '/*');
        $now = time();
        
        foreach ($files as $file) {
            if (is_file($file) && pathinfo($file, PATHINFO_EXTENSION) !== 'lock') {
                $poolData = json_decode(file_get_contents($file), true);
                
                if ($poolData) {
                    // Delete if expired or too old
                    if (isset($poolData['expire_at']) && $now > $poolData['expire_at']) {
                        @unlink($file);
                    } elseif (($now - $poolData['last_access']) > 300) {
                        // Delete if no access for 5 minutes
                        @unlink($file);
                    }
                }
            }
        }
    }
    
    // Get pool statistics
    public static function getPoolStats() {
        self::init();
        
        $files = glob(self::$poolDir . '/*');
        $totalUsers = 0;
        $activePools = 0;
        
        foreach ($files as $file) {
            if (is_file($file) && pathinfo($file, PATHINFO_EXTENSION) !== 'lock') {
                $poolData = json_decode(file_get_contents($file), true);
                
                if ($poolData && (time() - $poolData['last_access']) < 60) {
                    $activePools++;
                    $totalUsers += $poolData['users'];
                }
            }
        }
        
        return [
            'active_pools' => $activePools,
            'total_users' => $totalUsers
        ];
    }
}

// Clean old pools on script end
register_shutdown_function(function() {
    if (rand(1, 100) <= 10) { // 10% chance to clean
        StreamPool::cleanOldPools();
    }
});
