<?php
// Streaming Proxy Handler
require_once 'config.php';

// Get token from URL
$token = $_GET['token'] ?? '';
$type = $_GET['type'] ?? 'master'; // master or segment

if (empty($token)) {
    http_response_code(400);
    die('Invalid request');
}

// Get stream from database
$db = getDBConnection();
$stmt = $db->prepare("SELECT * FROM streams WHERE token = ? AND is_active = 1");
$stmt->execute([$token]);
$stream = $stmt->fetch();

if (!$stream) {
    http_response_code(404);
    die('Stream not found');
}

// Log access
if (getSetting('enable_logging') == '1') {
    logStreamAccess($stream['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT'] ?? '');
}

// Handle different types
if ($type === 'segment') {
    proxySegment($stream['source_url'], $_GET['segment'] ?? '');
} else {
    proxyMasterPlaylist($stream['source_url'], $token);
}

// Proxy master M3U8 playlist
function proxyMasterPlaylist($sourceUrl, $token) {
    // Fetch the master playlist
    $content = fetchUrl($sourceUrl);
    
    if ($content === false) {
        http_response_code(502);
        die('Failed to fetch stream');
    }
    
    // Parse and rewrite URLs in the playlist
    $baseUrl = dirname($sourceUrl);
    $lines = explode("\n", $content);
    $rewrittenContent = '';
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        if (empty($line) || $line[0] === '#') {
            // Keep comments and empty lines as-is
            $rewrittenContent .= $line . "\n";
        } else {
            // Rewrite URLs
            if (strpos($line, 'http') === 0) {
                // Absolute URL
                $segmentUrl = $line;
            } else {
                // Relative URL
                $segmentUrl = $baseUrl . '/' . $line;
            }
            
            // Check if it's a variant playlist or segment
            if (strpos($line, '.m3u8') !== false) {
                // It's a variant playlist, proxy it too
                $rewrittenUrl = SITE_URL . '/proxy.php?token=' . $token . '&type=variant&url=' . urlencode($segmentUrl);
            } else {
                // It's a segment
                $rewrittenUrl = SITE_URL . '/proxy.php?token=' . $token . '&type=segment&segment=' . urlencode($segmentUrl);
            }
            
            $rewrittenContent .= $rewrittenUrl . "\n";
        }
    }
    
    // Send headers
    header('Content-Type: application/vnd.apple.mpegurl');
    header('Access-Control-Allow-Origin: *');
    header('Cache-Control: no-cache');
    
    echo $rewrittenContent;
}

// Proxy video segments
function proxySegment($sourceUrl, $segmentPath) {
    if (empty($segmentPath)) {
        http_response_code(400);
        die('Invalid segment');
    }
    
    // Decode segment URL
    $segmentUrl = urldecode($segmentPath);
    
    // Check cache
    if (ENABLE_CACHE) {
        $cacheKey = md5($segmentUrl);
        $cacheFile = SEGMENT_CACHE_DIR . '/' . $cacheKey;
        
        if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < CACHE_LIFETIME) {
            // Serve from cache
            header('Content-Type: video/MP2T');
            header('Access-Control-Allow-Origin: *');
            header('X-Cache: HIT');
            readfile($cacheFile);
            return;
        }
    }
    
    // Fetch segment
    $ch = curl_init($segmentUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $content = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200 || $content === false) {
        http_response_code(502);
        die('Failed to fetch segment');
    }
    
    // Cache segment
    if (ENABLE_CACHE) {
        file_put_contents($cacheFile, $content);
    }
    
    // Send headers and content
    header('Content-Type: video/MP2T');
    header('Access-Control-Allow-Origin: *');
    header('X-Cache: MISS');
    header('Content-Length: ' . strlen($content));
    
    echo $content;
}

// Fetch URL content
function fetchUrl($url) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $content = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return ($httpCode === 200) ? $content : false;
}

// Log stream access
function logStreamAccess($streamId, $ip, $userAgent) {
    $db = getDBConnection();
    $stmt = $db->prepare("INSERT INTO stream_logs (stream_id, ip_address, user_agent) VALUES (?, ?, ?)");
    $stmt->execute([$streamId, $ip, $userAgent]);
}

// Get setting value
function getSetting($key) {
    $db = getDBConnection();
    $stmt = $db->prepare("SELECT setting_value FROM settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $result = $stmt->fetch();
    return $result ? $result['setting_value'] : null;
}
