<?php
// stream.php - Clean URL Stream Proxy
// Handles: /stream/TOKEN.m3u8
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once 'config.php';

// ============================================
// PARSE REQUEST
// ============================================

$requestUri = $_SERVER['REQUEST_URI'];
$path = parse_url($requestUri, PHP_URL_PATH);

// Extract token from URL path
preg_match('/\/(?:stream\/)?([a-f0-9]+)(?:\.m3u8)?/', $path, $matches);
$token = $matches[1] ?? '';

$type = $_GET['type'] ?? 'master';
$url = $_GET['url'] ?? '';
$segment = $_GET['segment'] ?? '';

// Validate token
if (empty($token)) {
    sendError(400, 'Invalid token');
}

// ============================================
// GET STREAM
// ============================================

$stream = getStreamByToken($token);
if (!$stream) {
    sendError(404, 'Stream not found or inactive');
}

// Log access
if (getSetting('enable_logging', '1') == '1') {
    logStreamAccess($stream['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT'] ?? '');
}

// ============================================
// ROUTE REQUEST
// ============================================

if ($type === 'segment' && !empty($segment)) {
    handleSegment($segment);
} elseif ($type === 'variant' && !empty($url)) {
    handleVariantPlaylist($url, $token);
} else {
    handleMasterPlaylist($stream['source_url'], $token);
}

// ============================================
// HANDLERS
// ============================================

function handleMasterPlaylist($sourceUrl, $token) {
    global $stream;
    
    // Try to get from shared pool first (bypass max user limit)
    $poolResult = StreamPool::getSharedStream($stream['id'], $sourceUrl, 'master');
    
    if ($poolResult['pooled']) {
        // Successfully got from pool
        $content = $poolResult['content'];
        
        // Add pool info header
        header('X-Pool-Users: ' . $poolResult['users']);
        header('X-Pool-Status: ' . (isset($poolResult['fresh']) ? 'FRESH' : 'SHARED'));
    } else {
        // Fallback to direct fetch
        $result = fetchUrl($sourceUrl);
        if (!$result['success']) {
            sendError(502, 'Failed to fetch source stream (HTTP ' . $result['http_code'] . ')');
        }
        $content = $result['content'];
    }
    
    // Process playlist
    $output = processM3U8Playlist($content, $sourceUrl, $token, 'master');
    
    // Send response
    sendM3U8Response($output);
}

function handleVariantPlaylist($variantUrl, $token) {
    global $stream;
    
    $variantUrl = urldecode($variantUrl);
    
    // Try to get from shared pool first
    $poolResult = StreamPool::getSharedStream($stream['id'], $variantUrl, 'variant');
    
    if ($poolResult['pooled']) {
        $content = $poolResult['content'];
        header('X-Pool-Users: ' . $poolResult['users']);
        header('X-Pool-Status: ' . (isset($poolResult['fresh']) ? 'FRESH' : 'SHARED'));
    } else {
        // Fallback to direct fetch
        $result = fetchUrl($variantUrl);
        if (!$result['success']) {
            sendError(502, 'Failed to fetch variant playlist');
        }
        $content = $result['content'];
    }
    
    // Process playlist
    $output = processM3U8Playlist($content, $variantUrl, $token, 'variant');
    
    // Send response
    sendM3U8Response($output);
}

function handleSegment($segmentPath) {
    $segmentUrl = urldecode($segmentPath);
    
    // Check cache
    $cached = getCachedContent($segmentUrl);
    if ($cached !== false) {
        $contentType = getContentType($segmentUrl);
        sendSegmentResponse($cached, $contentType, true);
    }
    
    // Fetch segment
    $result = fetchUrl($segmentUrl);
    if (!$result['success']) {
        sendError(502, 'Failed to fetch segment (HTTP ' . $result['http_code'] . ')');
    }
    
    // Cache segment
    setCachedContent($segmentUrl, $result['content']);
    
    // Determine content type
    $contentType = $result['content_type'] ?: getContentType($segmentUrl);
    
    // Send response
    sendSegmentResponse($result['content'], $contentType, false);
}
